<?php

namespace App\Http\Controllers;

use App\Models\Campaign;
use App\Models\ProofSubmission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ProofSubmissionController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        if (!$user->isPromoter()) {
            abort(403);
        }

        $submissions = $user->submissions()->with('campaign')->latest()->paginate(10);
        return view('submissions.index', compact('submissions'));
    }

    public function store(Request $request, Campaign $campaign)
    {
        $user = auth()->user();

        if (!$user->isPromoter()) {
            abort(403);
        }

        $request->validate([
            'screenshot' => 'required|image|max:5120', // 5MB max
            'member_growth' => 'required|integer|min:0',
        ]);

        // Check for cooldown (e.g., once per 24 hours per campaign)
        $lastSubmission = $user->submissions()
            ->where('campaign_id', $campaign->id)
            ->where('created_at', '>', now()->subHours(24))
            ->first();

        if ($lastSubmission) {
            return back()->with('error', 'You can only submit proof for the same campaign once every 24 hours.');
        }

        $path = $request->file('screenshot')->store('proofs', 'public');

        $campaign->submissions()->create([
            'promoter_id' => $user->id,
            'screenshot_path' => $path,
            'verification_code' => strtoupper(substr($user->id, 0, 8)), // Auto-generate
            'member_growth' => $request->member_growth,
            'status' => 'pending',
        ]);

        return redirect()->route('campaigns.browse')->with('success', 'Proof submitted successfully and is awaiting admin review.');
    }
}
